// Log temperature and depth
//
// LogTD is copyright (c) 2000, 2012 by
// Paul Gettings, Dep't of Geology & Geophysics, University of Utah
// All Rights Reserved.
// 
// LogTD is free software; you can redistribute it and/or modify it
// under the terms of the GNU General Public License, version 2, as
// published by the Free Software Foundation.
// 
// LogTD is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details, in the file
// COPYING.  If that file is not present, write to the Free
// Software Foundation, 59 Temple Place - Suite 330, Boston, MA  
// 02111-1307, USA
// 
// Please forward any improvements to <p.gettings@utah.edu>.
//

import java.util.*;
import java.io.*;
import java.text.*;
import Utah.DAS.Serial.*;
import Utah.util.*;

/** Log temperature and depth for a borehole probe.
    Uses SerialDAS as the interface to the 2 "instruments" - a DP100
    multimeter and a 8033 Counter.  Both instruments are read on demand
    devices, so we send a command, and then wait for the response.
  */
public class LogTD {
  private static SerialDAS das;

  private static String empty = new String();

  private static int Delay = 500;

  private static int FailSafeAbort = 40;

  private static boolean MMflag = true;
  private static boolean DCflag = true;

  private static PrintWriter out;
  private static String multimeterPortName, counterPortName;
  private static int multimeterBaud=9600, counterBaud=9600;
  private static boolean debug=false;

  private static double DepthConstant=1.0;
  private static double DepthError=0.1; // meters

  private static boolean run = true;

  private static String VersionString = "v1.2.0";
  private static double depthOffset=0.0;

  private static double calA=-3.7993, calB=4357.2, calC=309.56;

  public static void main(String[] args) {
    LineNumberReader stdin;
    double resistance=0.0, depth=0.0, oldDepth=0.0;

    long start, startRead, endRead;

    stdin = new LineNumberReader(new InputStreamReader(System.in));

    System.out.println("****************************************");
    System.out.println("** Temperature vs. Depth Logging Program");
    System.out.println("** "+VersionString);
    System.out.println("** Paul Gettings");
    System.out.println("** Dep't of Geology & Geophysics");
    System.out.println("** University of Utah");
    System.out.println("****************************************");
    System.out.println("");

    // get ports, etc. from config file
    dprint("Reading configuration file");
    Hashtable tokens = ParseConfigFile.grok("logtd.conf");
    grokHashtable(tokens);

    // get the output filename
    System.out.print("Output filename: ");
    String filename=null;
    try {
      filename = stdin.readLine();
      filename = filename.trim();
      }
    catch (IOException _) {
      System.out.println("I/O error reading filename from stdin.");
      System.exit(1);
      }
    
    System.out.print("Depth offset, in meters: ");
    String line="";
    try {
      line = stdin.readLine();
      line = line.trim();
      }
    catch (IOException _) {
      System.out.println("I/O error reading depth offset from stdin.");
      System.exit(1);
      }

    try {
      depthOffset = Double.parseDouble(line);
      }
    catch (NumberFormatException _) {
      System.out.println("Error converting entered depth offset to a number.  Using 0.0.");
      depthOffset = 0.0;
      }

    System.out.print("Comment: ");
    String comment="";
    try {
      comment = stdin.readLine();
      comment = comment.trim();
      }
    catch (IOException _) {
      System.out.println("I/O error reading comment from stdin.");
      System.exit(1);
      }

    // open the output file
    try {
      out = new PrintWriter(new FileWriter(filename));
      }
    catch (IOException _) {
      System.out.println("I/O error opening file '"+filename+"'. Abort.");
      System.exit(1);
      }

    // fire up the DAS
    System.out.println("\nStarting the DAS.");
    das = new SerialDAS();

    // open up the ports
    // Multimeter is instrument 0
    char[] recordChars = { '\0', '\n'};
    if(MMflag) {
      if(!das.openPort(0, multimeterPortName, multimeterBaud, 8, 1, 0,
			  recordChars, 0, "", debug)) {
	System.out.println("Error opening port '"+multimeterPortName+
			   "' for multimeter.");
	System.exit(1);
	}
      }
    // Counter is instrument 1
    if(DCflag) {
      recordChars[0] = '\0'; recordChars[1] = '\r';
      if(!das.openPort(1, counterPortName, counterBaud, 8, 1, 0,
			  recordChars, 0, "", debug)) {
	System.out.println("Error opening port '"+multimeterPortName+
			   "' for multimeter.");
	System.exit(1);
	}
      }

    // start our instrument read thread
    dprint("Starting data acquisition threads.");
    das.start();

    // setup instruments
    if(MMflag) {
      dprint("Setting up the multimeter.");
      setupMultimeter();
      }
    if(DCflag) {
      dprint("Setting up the counter.");
      setupCounter();
      }

    // write data header to file
    out.println("#Created by LogTD, "+VersionString);
    out.println("#Logging Parameters");
    out.println("#Logging date: "+DateFormat.getDateTimeInstance().format(new Date()));
    out.println("#Output file: "+filename);
    out.println("#Multimeter in use? "+MMflag);
    out.println("#Depth counter in use? "+DCflag);
    out.println("#Multimeter data port: "+multimeterPortName);
    out.println("#Counter data port:    "+counterPortName);
    out.println("#Multimeter baud rate: "+multimeterBaud);
    out.println("#Counter baud rate:    "+counterBaud);
    out.println("#Debug flag is "+debug);
    out.println("#");
    out.println("#Depth factor: "+DepthConstant+" meters/count");
    out.println("#Depth offset: "+depthOffset+" meters");
    out.println("#Read Delay: "+Delay+" milliseconds");
    out.println("#");
    out.println("#Comment:"+comment);
    out.println("#");

    // start reading until we die
    start = System.currentTimeMillis();
    double time, timeAtDepth=0, startTimeAtDepth=0;
    dprint("Entering main loop.");
    System.out.println("Logging starting.  End by typing 'Q' with a return.");
    System.out.println("Output format is:");
    System.out.println(" Resistance\t   Probe  \t Elapsed  \tEstimated\tTime at");
    System.out.println("   (Ohms)  \t Depth (m)\tTime(secs)\t Temp(C) \t Depth");
    out.println("#Resistance\t   Probe  \t Elapsed  ");
    out.println("#  (Ohms)  \t Depth (m)\tTime(secs)");
    long count=0;
    while(run) {
      // DEBUG
      //if(count > 10) run=false;
      count++;
      // END DEBUG
      startRead = System.currentTimeMillis();
      if(MMflag) {
	dprint("  Read resistance.");
	resistance = getResistance();
	}
      if(DCflag) {
	dprint("  Read depth.");
	depth = getDepth();
	}
      endRead = System.currentTimeMillis();
      // Compute the midpoint time of the readings, and shift to ms
      // relative to start of logging.  Then rescale to seconds, from
      // ms
      time = ( (startRead + endRead) - (2*start))/2000.0;
      // Check if holding at a depth; if so, compute time at depth
      if(Math.abs(depth - oldDepth) < DepthError) {
        timeAtDepth = time - startTimeAtDepth;
        }
      else {
        startTimeAtDepth = time;
        timeAtDepth = 0;
	oldDepth = depth;
        }
      // write the data to disk
      dprint ("  write record.");
      writeRecord(resistance, depth, time, timeAtDepth);
      // check for keyboard input
      try {
	dprint("  check keyboard buffer.");
	if(System.in.available() > 0) {
	  checkKeyboard();
	  }
	}
      catch (IOException _) { };
      // pause for a bit
      try {
	Thread.sleep(Delay);
	}
      catch (InterruptedException _) { }
      }

    // shut it all down
    System.out.println("Shutdown.");
    System.out.println(count+" records read");
    das.end();
    try {
      Thread.sleep(1000);
      }
    catch (InterruptedException _) { }
    out.close();
    }

  /** Setup the multimeter for reading.
    * Panel meter set to continuously output readings; so no setup
    * needed...
    */
  public static void setupMultimeter() {
    // 
    System.out.println("Setup multimeter.");
    das.flushInput(0);
    System.out.println("  done.");
    }

  /** Setup the counter for reading.
    * Enable the counter address, set no linefeeds, start counting
    */
  public static void setupCounter() {
    System.out.println("Setup depth counter.");
    das.flushInput(1);
    // enable the counter's address (so it will talk)
    System.out.println("  Enable address.");
    das.write(1, "AE01\r");
    waitForPrompt(1, "HELLO");
    // disable linefeeds after responses
    System.out.println("  Disable linefeeds.");
    das.write(1, "LF0\r");
    waitForPrompt(1, "OK");
    // Set the count to some depth
    System.out.println("  Set starting count.");
    int count;
    count = (int)(depthOffset/DepthConstant) + 1;
    das.write(1, "PC"+count+"\r");
    waitForPrompt(1, "OK");
    // Start the counter
    System.out.println("  Start counting.");
    das.write(1, "ST\r");
    }

  /** Get current resistance.
    * Gets the current resistance measurement from the
    * multimeter.  No conversion is done.  Result is in
    * Ohms.
    */
  public static double getResistance() {
    String rec = null;
    // clear any waiting records
    while((rec=das.read(0)) != null );

    // wait for the next record
    int count=0;
    while(rec == null ) {
      rec = das.read(0);
      if(rec != null) {
        rec = rec.trim();
	if(rec.endsWith("A")) {
	  // No alarms, no overload
	  continue;
	  }
	if(rec.endsWith("E")) {
	  // Overload, no alarms
	  System.out.println("  Overload condition reported");
	  continue;
	  }
	if(rec.equals("")) {
	  // empty string; something funky happened here
	  rec = null;
	  }
	}
      // fail-safe
      if(count > FailSafeAbort) {
        System.out.println("Fail-safe abort of resistance read:");
        System.out.println("  Set R to 0.0; flush input on device");
        das.flushInput(0);
        rec = "0.0";
        }
      count++;

      // sleep for a bit
      try {
	Thread.sleep(50);
	}
      catch (InterruptedException _) { }
      }

    double resistance;
    try {
      resistance = Double.parseDouble(rec.trim().substring(0,rec.indexOf(".")));
      }
    catch (NumberFormatException _) {
      System.out.println("Bad resistance record: ["+rec+"]");
      resistance=0.0;
      }
    return(resistance);
    }

  /** Get the current depth, in meters.
    * Reads the depth counter, and converts to meters.
    * Assumes the correct depth constant is set somewhere
    */
  public static double getDepth() {
    String rec = null;
    // clear any waiting records
    while((rec=das.read(1)) != null );

    // request current depth count
    das.write(1, "RD\r");

    // wait for the next record
    int count = 0;
    while(rec == null ) {
      rec = das.read(1);
      // fail-safe
      if(count > FailSafeAbort) {
        System.out.println("Fail-safe abort of depth read: Depth count set to -1.");
        rec = "-1";
        das.flushInput(1);
        }
      count++;
      // sleep for a bit
      try {
	Thread.sleep(50);
	}
      catch (InterruptedException _) { }
      }

    // parse the depth count
    long depthCount;
    try {
      depthCount = Long.parseLong(rec.trim());
      }
    catch (NumberFormatException _) {
      System.out.println("Bad depth record: ["+rec+"]");
      depthCount=0;
      }

    // convert to depth in meters
    double depth;
    depth = depthCount * DepthConstant; // meters/count
    return(depth);
    }

  /** Wait for a specific prompt string.
    * Waits for the prompt string "prompt" from instrument num.
    * The prompt is trimmed of whitespace.
    * Each loop has a 50 ms sleep, and up to 20 records are checked.
    */
  public static void waitForPrompt(int num, String prompt) {
    String rec;
    int i=0;
    while(i < 20) {
      int j = 0;
      while((rec=das.read(num)) == null) {
	try {
	  Thread.sleep(50);
	  }
	catch (InterruptedException _) { }
	if(j > FailSafeAbort) {
	  System.out.println("Fail-safe abort of waitForPrompt; instrument "+num+" prompt "+prompt+"\n");
	  i=21;
	  return;
	  }
	j++;
	}
      dprint("waitForPrompt: ["+rec+"]\n");
      if((rec.trim()).endsWith(prompt.trim())) {
	break;
	}
      i++;
      }
    }


  /** Given a Hashtable of token/value pairs from the configuration
    * file, set the class variables of interest.
    */
  public static void grokHashtable(Hashtable T) {
    Enumeration keys = T.keys();
    String token, value;

    while(keys.hasMoreElements()) {
      token = (String)keys.nextElement();
      value = (String) T.get(token);

      // see if we recognize the token
      if(token.toUpperCase().equals("MMPORT")) {
        multimeterPortName = value;
        }
      else if(token.toUpperCase().equals("COUNTERPORT")) {
        counterPortName = value;
        }
      else if(token.toUpperCase().equals("MMBAUD")) {
        multimeterBaud = Integer.parseInt(value);
        }
      else if(token.toUpperCase().equals("COUNTERBAUD")) {
        counterBaud = Integer.parseInt(value);
        }
      else if(token.toUpperCase().equals("DEPTHCONSTANT")) {
        DepthConstant = Double.parseDouble(value);
        }
      else if(token.toUpperCase().equals("DEPTHERROR")) { 
        DepthError = Double.parseDouble(value);
        }
      else if(token.toUpperCase().equals("DELAY")) {
        Delay = Integer.parseInt(value);
        }
      else if(token.toUpperCase().equals("DEBUG")) {
        debug = true;
        }
      else if(token.toUpperCase().equals("FAILSAFE")) {
        FailSafeAbort = Integer.parseInt(value);
        }
      else if(token.toUpperCase().equals("CONSTANTA")) {
        calA = Double.parseDouble(value);
        }
      else if(token.toUpperCase().equals("CONSTANTB")) {
        calB = Double.parseDouble(value);
        }
      else if(token.toUpperCase().equals("CONSTANTC")) {
        calC = Double.parseDouble(value);
        }
      else if(token.toUpperCase().equals("MM")) {
        if(value.toUpperCase().equals("FALSE")) {
	  MMflag = false;
	  }
	else {
	  MMflag = true;
	  }
	}
      else if(token.toUpperCase().equals("DC")) {
        if(value.toUpperCase().equals("FALSE")) {
	  DCflag = false;
	  }
	else {
	  DCflag = true;
	  }
        }

      else {
        System.out.println("Unknown configuration token '"+token+"'");
        }
      }
    }

  private static final DecimalFormat resistFmt = new DecimalFormat("  000000.0");
  private static final DecimalFormat tempFmt = new DecimalFormat("000.000");
  private static final DecimalFormat depthFmt = new DecimalFormat("0000.00000");
  private static final DecimalFormat timeFmt = new DecimalFormat("000000.000");
  private static final DecimalFormat timeFmt2 = new DecimalFormat("0000.0");
  private static String R, D, T, outString;

  public static void writeRecord(double resistance, double depth,
  				double time, double timeatdepth) {
    double t;
    R = resistFmt.format(resistance);
    D = depthFmt.format(depth);
    T = timeFmt.format(time);
    outString = " "+R+"\t"+D+"\t"+T;
    t = calB/(Math.log(resistance) - calA) - calC;
    outString += "\t"+tempFmt.format(t);
    out.println(outString);
    // screen display also has time spent at current depth
    outString += " \t"+timeFmt2.format(timeatdepth);
    System.out.println(outString);
    }

  public static void checkKeyboard() {
    int nB;
    try {
      if((nB=System.in.available()) < 1) {
	return;
	}
      byte[] buf = new byte[nB];
      nB = System.in.read(buf, 0, buf.length);
      for(int i=0; i<nB; i++) {
	if(buf[i] == 'Q') {
	  run = false;
	  dprint("checkKeyboard: Q pressed");
	  }
	}
      }
    catch (IOException _) { }
    return;
    }

  public static void dprint(String msg) {
    if(debug) {
      System.err.println(msg);
      }
    }

}// end class LogTD
